// Nutrition Consultations - Version 4 (HTML/CSS/JS) - Dashboard Style

let currentSlide = 0
let slides = []
let dots = []
let slideInterval
let selectedGoal = ''
let consultations = []

const planIcons = {
  'Сбалансированное питание': '<svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M17.5 12c0 4.5-2.5 6.5-5 8.5-2.5-2-5-4-5-8.5 0-2.5 2.5-5 5-5s5 2.5 5 5z"/><path d="M12 7v10"/><path d="M7 12h10"/></svg>',
  'Детокс программа': '<svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M20.84 4.61a5.5 5.5 0 0 0-7.78 0L12 5.67l-1.06-1.06a5.5 5.5 0 0 0-7.78 7.78l1.06 1.06L12 21.23l7.78-7.78 1.06-1.06a5.5 5.5 0 0 0 0-7.78z"/></svg>',
  'План для набора массы': '<svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="23 6 13.5 15.5 8.5 10.5 1 18"/><polyline points="17 6 23 6 23 12"/></svg>',
  'Аллергии': '<svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"/></svg>'
}

function getPlanIcon(planName) {
  for (const [key, icon] of Object.entries(planIcons)) {
    if (planName.toLowerCase().includes(key.toLowerCase().split(' ')[0])) {
      return icon
    }
  }
  return planIcons['Сбалансированное питание']
}

function initPage() {
  const currentPage = window.location.pathname.includes('enroll.html') ? 'enroll' : 'index'
  
  if (currentPage === 'enroll') {
    initEnrollPage()
  } else {
    initIndexPage()
  }
  
  initSlider()
}

function initIndexPage() {
  loadConsultations()
  updateStats()
  renderProgressBars()
  
  setInterval(() => {
    loadConsultations()
    updateStats()
    renderProgressBars()
  }, 500)
}

function loadConsultations() {
  const saved = localStorage.getItem('nutriConsultations')
  
  if (saved) {
    try {
      consultations = JSON.parse(saved)
    } catch (e) {
      consultations = []
    }
  } else {
    consultations = [
      {
        id: 1,
        plan: 'Сбалансированное питание',
        date: '2025-01-20',
        rating: 5,
        photo: 'images/plan-1.jpg'
      },
      {
        id: 2,
        plan: 'Детокс программа',
        date: '2025-01-10',
        rating: 4,
        photo: 'images/plan-2.jpg'
      },
      {
        id: 3,
        plan: 'План для набора массы',
        date: '2024-12-28',
        rating: 5,
        photo: 'images/plan-3.jpg'
      }
    ]
    localStorage.setItem('nutriConsultations', JSON.stringify(consultations))
  }
  
  renderConsultations()
}

function renderConsultations() {
  const container = document.getElementById('consultations-grid')
  if (!container) return
  
  if (consultations.length === 0) {
    container.innerHTML = `
      <div class="empty-dashboard">
        <div class="empty-icon-dashboard">
          <svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <polyline points="22 12 18 12 15 21 9 3 6 12 2 12"/>
          </svg>
        </div>
        <h2 class="empty-title-dashboard">Нет консультаций</h2>
        <p class="empty-text-dashboard">Начните с первой консультации</p>
        <a href="enroll.html" class="empty-button-dashboard">Записаться</a>
      </div>
    `
    return
  }
  
  container.innerHTML = consultations.map(consultation => {
    const date = new Date(consultation.date).toLocaleDateString('ru-RU', {
      day: 'numeric',
      month: 'short',
      year: 'numeric'
    })
    
    const icon = getPlanIcon(consultation.plan)
    
    return `
      <div class="consultation-card-dashboard">
        <div class="card-header-dashboard">
          <div class="card-image-dashboard">
            <img src="${consultation.photo}" alt="${consultation.plan}">
            <div class="card-icon-overlay">
              <div class="card-plan-icon">
                ${icon}
              </div>
            </div>
            <div class="card-status-badge">
              <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <polyline points="13 3 13 10 19 10 11 21 11 14 5 14 13 3"/>
              </svg>
              <span>Активен</span>
            </div>
          </div>
        </div>
        <div class="card-body-dashboard">
          <h3 class="card-title-dashboard">${consultation.plan}</h3>
          <div class="card-meta-dashboard">
            <div class="meta-item-dashboard">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <rect x="3" y="4" width="18" height="18" rx="2" ry="2"/>
                <line x1="16" y1="2" x2="16" y2="6"/>
                <line x1="8" y1="2" x2="8" y2="6"/>
                <line x1="3" y1="10" x2="21" y2="10"/>
              </svg>
              <span>${date}</span>
            </div>
            <div class="meta-item-dashboard rating-dashboard">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <polygon points="12 2 15.09 8.26 22 9.27 17 14.14 18.18 21.02 12 17.77 5.82 21.02 7 14.14 2 9.27 8.91 8.26 12 2"/>
              </svg>
              <span>${consultation.rating}/5</span>
            </div>
          </div>
        </div>
      </div>
    `
  }).join('')
}

function updateStats() {
  const statPlans = document.getElementById('stat-plans')
  const statRating = document.getElementById('stat-rating')
  const statExcellent = document.getElementById('stat-excellent')
  const sectionCount = document.getElementById('section-count')
  
  if (statPlans) {
    statPlans.textContent = consultations.length
  }
  
  if (statRating) {
    const avgRating = consultations.length > 0 
      ? (consultations.reduce((sum, c) => sum + (c.rating || 0), 0) / consultations.length).toFixed(1)
      : '0.0'
    statRating.textContent = avgRating
  }
  
  if (statExcellent) {
    statExcellent.textContent = consultations.filter(c => c.rating === 5).length
  }
  
  if (sectionCount) {
    sectionCount.textContent = `${consultations.length} планов`
  }
}

function renderProgressBars() {
  const container = document.getElementById('progress-bars')
  if (!container) return
  
  if (consultations.length === 0) {
    container.innerHTML = '<p style="color: var(--text-secondary); text-align: center;">Нет данных для отображения</p>'
    return
  }
  
  container.innerHTML = consultations.map(consultation => {
    const percentage = (consultation.rating / 5) * 100
    const shortName = consultation.plan.length > 15 
      ? consultation.plan.substring(0, 15) + '...'
      : consultation.plan
    
    return `
      <div class="progress-bar-item">
        <div class="progress-bar-label">${shortName}</div>
        <div class="progress-bar-wrapper">
          <div class="progress-bar-fill" style="width: ${percentage}%"></div>
        </div>
        <div class="progress-bar-value">${consultation.rating}/5</div>
      </div>
    `
  }).join('')
  
  // Animate progress bars
  setTimeout(() => {
    const fills = container.querySelectorAll('.progress-bar-fill')
    fills.forEach(fill => {
      const width = fill.style.width
      fill.style.width = '0%'
      setTimeout(() => {
        fill.style.width = width
      }, 100)
    })
  }, 100)
}

function initEnrollPage() {
  const goalButtons = document.querySelectorAll('.goal-card-dashboard')
  const dateGroup = document.getElementById('date-group')
  const dateInput = document.getElementById('consultation-date')
  const submitBtn = document.getElementById('submit-btn')
  const form = document.getElementById('enroll-form')
  const successMessage = document.getElementById('success-message')
  
  // Set min date to today
  const today = new Date()
  const maxDate = new Date(today.getTime() + 180 * 24 * 60 * 60 * 1000)
  const minDateStr = today.toISOString().split('T')[0]
  const maxDateStr = maxDate.toISOString().split('T')[0]
  
  if (dateInput) {
    dateInput.min = minDateStr
    dateInput.max = maxDateStr
  }
  
  goalButtons.forEach(button => {
    button.addEventListener('click', () => {
      goalButtons.forEach(btn => btn.classList.remove('selected'))
      button.classList.add('selected')
      selectedGoal = button.dataset.goal
      
      if (dateGroup) {
        dateGroup.style.display = 'flex'
        dateGroup.style.flexDirection = 'column'
      }
      
      if (submitBtn) {
        submitBtn.disabled = false
      }
    })
  })
  
  if (dateInput) {
    dateInput.addEventListener('change', () => {
      if (dateInput.value && selectedGoal) {
        if (submitBtn) {
          submitBtn.disabled = false
        }
      }
    })
  }
  
  if (form) {
    form.addEventListener('submit', (e) => {
      e.preventDefault()
      
      if (!selectedGoal || !dateInput.value) return
      
      const goalNames = {
        'weight-loss': 'Похудение',
        'weight-gain': 'Набор массы',
        'healthy-lifestyle': 'ЗОЖ',
        'allergies': 'Аллергии'
      }
      
      const saved = localStorage.getItem('nutriConsultations')
      const consultations = saved ? JSON.parse(saved) : []
      
      const newConsultation = {
        id: Date.now(),
        plan: `План: ${goalNames[selectedGoal]}`,
        date: dateInput.value,
        rating: null,
        photo: 'images/plan-placeholder.jpg'
      }
      
      consultations.push(newConsultation)
      localStorage.setItem('nutriConsultations', JSON.stringify(consultations))
      
      if (successMessage) {
        successMessage.style.display = 'flex'
      }
      
      setTimeout(() => {
        window.location.href = 'index.html'
      }, 2000)
    })
  }
}

function initSlider() {
  slides = document.querySelectorAll('.slide-dashboard')
  dots = document.querySelectorAll('.dot-dashboard')
  const prevBtn = document.querySelector('.slider-btn-prev-dashboard')
  const nextBtn = document.querySelector('.slider-btn-next-dashboard')
  
  if (slides.length === 0) return
  
  function showSlide(index) {
    slides.forEach((slide, i) => {
      slide.classList.toggle('active', i === index)
    })
    
    dots.forEach((dot, i) => {
      dot.classList.toggle('active', i === index)
    })
    
    currentSlide = index
  }
  
  function nextSlide() {
    const next = (currentSlide + 1) % slides.length
    showSlide(next)
  }
  
  function prevSlide() {
    const prev = (currentSlide - 1 + slides.length) % slides.length
    showSlide(prev)
  }
  
  if (prevBtn) {
    prevBtn.addEventListener('click', prevSlide)
  }
  
  if (nextBtn) {
    nextBtn.addEventListener('click', nextSlide)
  }
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => showSlide(index))
  })
  
  slideInterval = setInterval(nextSlide, 3000)
  
  // Pause on hover
  const slider = document.querySelector('.slider-container-dashboard')
  if (slider) {
    slider.addEventListener('mouseenter', () => clearInterval(slideInterval))
    slider.addEventListener('mouseleave', () => {
      slideInterval = setInterval(nextSlide, 3000)
    })
  }
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', initPage)

